#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>

#include "question.h"
#include "buffer.h"
#include "record.h"

void create_question(const char* domain, RecordType qtype, Question* question) {
    uint8_t len = strlen(domain);
    question->domain = malloc(len + 1);
    question->qtype = qtype;
    question->domain[0] = len;
    question->cls = 1;
    memcpy(question->domain + 1, domain, len);
}

bool read_question(PacketBuffer* buffer, Question* question) {
    buffer_read_qname(buffer, &question->domain);
    
    uint16_t qtype_num = buffer_read_short(buffer);
    record_from_id(qtype_num, &question->qtype);
    question->cls = buffer_read_short(buffer);

    if (question->qtype == UNKOWN) {
        free(question->domain);
        return false;
    }

    return true;
}

void write_question(PacketBuffer* buffer, Question* question) {
    buffer_write_qname(buffer, question->domain);
    
    uint16_t id = record_to_id(question->qtype);
    buffer_write_short(buffer, id);

    buffer_write_short(buffer, question->cls);
}

void free_question(Question* question) {
    free(question->domain);
}

static const char* class_to_str(Question* question) {
    switch(question->cls) {
        case 1: return "IN";
        case 3: return "CH";
        case 4: return "HS";
        default: return "??";
    }
}

static const char* qtype_to_str(Question* question) {
    switch(question->qtype) {
        case A: return "A";
        case NS: return "NS";
        case CNAME: return "CNAME";
        case SOA: return "SOA";
        case PTR: return "PTR";
        case MX: return "MX";
        case TXT: return "TXT";
        case AAAA: return "AAAA";
        case SRV: return "SRV";
        case CAA: return "CAA";
        default: return "UNKOWN";
    }
}

void print_question(Question* question) {
    printf("%.*s.\t%s %s\n",
        question->domain[0],
        question->domain + 1,
        class_to_str(question),
        qtype_to_str(question)
    );
}
