#include "idt.h"
#include "io.h"
#include "pic.h"
#include "../panic.h"
#include "../keyboard.h"
#include <stddef.h>
#include <stdint.h>
#include <stdbool.h>

void idt_pic_timer(uint8_t exception) {
    pic_eoi(exception - PIC_REMAP_OFFSET);
}

void idt_pic_keyboard(uint8_t exception) {
    pic_eoi(exception - PIC_REMAP_OFFSET);
    uint8_t c = inb(0x60);
    if(c == 0xe0) {
        set_next_extended();
    } else if(c >= 0x80) {
        process_keycode(c - 0x80, true);
    } else {
        process_keycode(c, false);
    }
}

void idt_pic_handler(uint8_t exception) {
    pic_eoi(exception - PIC_REMAP_OFFSET);
    exception -= PIC_REMAP_OFFSET;
    panic("IRQ: 0x%x", exception);
}

void idt_exception_handler(uint8_t exception) {
    switch(exception) {
        case 0x00:
            panic("Div by zero");
            break;
        case 0x08:
            panic("Double fault");
            break;
        default:
            panic("Error: 0x%x", exception);
            break;
    }
}

void idt_set_descriptor(uint8_t vector, void* isr, uint8_t flags) {
    IDTEntry* descriptor = &idt[vector];
 
    descriptor->isr_low = (size_t)isr & 0xffff;
    descriptor->kernel_code_selector = 0x08; // Kernel code offset in GDT
    descriptor->attributes = flags;
    descriptor->isr_high = (size_t)isr >> 16;
    descriptor->reserved = 0;
}

void idt_enable_gate(uint8_t vector) {
    idt[vector].attributes |= IDT_FLAG_PRESENT;
}

void idt_disable_gate(uint8_t vector) {
    idt[vector].attributes &= ~IDT_FLAG_PRESENT;
}

void idt_init() {
    idtr.base = (uintptr_t)&idt[0];
    idtr.limit = (uint16_t)sizeof(IDTEntry) * IDT_SIZE - 1;
 
    for (uint8_t vector = 0; vector < IDT_INTERRUPTS; vector++) {
        idt_set_descriptor(vector, isr_stub_table[vector], 0x8e);
    }
 
    __asm__ volatile ("lidt %0" : : "m"(idtr)); // load the new IDT
    __asm__ volatile ("sti"); // set the interrupt flag
}
