#include "stdio.h"
#include "string.h"
#include "util.h"
#include <stddef.h>
#include <stdarg.h>
#include <stdint.h>


int p_line = 0;
int p_column = 0;
uint8_t p_color = 0x07;

size_t screenaddr(int x, int y) {
    return VGA_VIDEO_MEMORY + 2*x + 2*SCREEN_WIDTH*y;
}

void writec(int x, int y, char c, char color) {
    size_t addr = screenaddr(x, y);
    *(char*)(addr) = c;
    *(char*)(addr + 1) = color;
}

void write(int x, int y, const char* str, char color) {
    while(*str != 0) {
        writec(x, y, *str, color);
        str++;
        x++;
    }
}

void scroll(int lines) {
    for(int i = lines; i < SCREEN_HEIGHT; i++) {
        void* src = (void*)screenaddr(0, i);
        void* dst = (void*)screenaddr(0, i-lines);
        memcpy(dst, src, 2*SCREEN_WIDTH);
    }
    for(int i = SCREEN_HEIGHT - lines; i < SCREEN_HEIGHT; i++) {
        clearline(i);
    }
}

void clearline(int line) {
    void* dst = (void*)screenaddr(0, line);
    memset(dst, 0, 2*SCREEN_WIDTH);
}

void clearscreen() {
    for(int i = 0; i < SCREEN_HEIGHT; i++) {
        void* dst = (void*)screenaddr(0, i);
        memset(dst, 0, 2*SCREEN_WIDTH);
    }
}

void nextline() {
    if(p_line >= SCREEN_HEIGHT - 1) {
        scroll(1);
    } else {
        p_line += 1;
    }
    p_column = 0;
}

void setcol(char col) {
    p_color = col;
}

void resetcol(void) {
    p_color = F_WHITE;
}

void putc(char c) {
    if(c == '\n') {
        nextline();
    } else {
        writec(p_column, p_line, c, p_color);
        p_column++;
        if(p_column >= SCREEN_WIDTH) {
            nextline();
        }
    }
}

void print(const char* str) {
    while(*str != 0) {
        putc(*str);
        str++;
    }
}

void println(const char* str) {
    print(str);
    nextline();
}

void printfv(const char* fstr, va_list args) {
    while(*fstr != 0) {
        if(*fstr == '%') {
            fstr++;
            char buf[20];
            switch(*fstr) {
                case '%':
                    putc('%');
                    break;
                case 's':
                    print(va_arg(args, char*));
                    break;
                case 'd':
                    print(itoa(va_arg(args, int), buf, 10));
                    break;
                case 'u':
                    print(itoa(va_arg(args, unsigned int), buf, 10));
                    break;
                case 'x':
                    print(itoa(va_arg(args, unsigned int), buf, 16));
                    break;
                case 'p':
                    print(itoa((size_t)(va_arg(args, void*)), buf, 16));
                    break;
                case '$': {
                    uint8_t msn = hex2i(*++fstr);
                    uint8_t lsn = hex2i(*++fstr);
                    p_color = (msn << 4) | lsn;
                    break;
                }
                default:
                    break;
            }
        } else {
            putc(*fstr);
        }
        fstr++;
    }
}

void printf(const char* fstr, ...) {
    va_list args;
    va_start(args, fstr);
    printfv(fstr, args);
}

void printfln(const char* fstr, ...) {
    va_list args;
    va_start(args, fstr);
    printfv(fstr, args);
    nextline();
}
